<?php

class CloudflareAPI
{
    private string $email;
    private string $apiKey;
    private string $baseUrl = "https://api.cloudflare.com/client/v4";

    public function __construct(string $email, string $apiKey)
    {
        $this->email = $email;
        $this->apiKey = $apiKey;
    }

    private function request(string $method, string $endpoint, array $data = [])
    {
        $curl = curl_init();

        $options = [
            CURLOPT_URL => $this->baseUrl . $endpoint,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => strtoupper($method),
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_HTTPHEADER => [
                "X-Auth-Email: " . $this->email,
                "X-Auth-Key: " . $this->apiKey,
                "Content-Type: application/json"
            ]
        ];

        if (count($data) > 0) {
            $options[CURLOPT_POSTFIELDS] = json_encode($data);
        }

        curl_setopt_array($curl, $options);

        $response = curl_exec($curl);
        $error = curl_error($curl);
        curl_close($curl);

        if ($error) {
            return [
                "success" => false,
                "error" => $error
            ];
        }

        return json_decode($response, true);
    }

    /**
     * ================================
     *  Zones
     * ================================
     */
    public function listZones()
    {
        return $this->request("GET", "/zones");
    }

    public function getZoneId(string $domain)
    {
        $zones = $this->listZones();

        if (!empty($zones['result'])) {
            foreach ($zones['result'] as $zone) {
                if ($zone['name'] === $domain) {
                    return $zone['id'];
                }
            }
        }

        return null;
    }

    /**
     * ================================
     *  DNS Records
     * ================================
     */
    public function listRecords(string $zoneId)
    {
        return $this->request("GET", "/zones/$zoneId/dns_records");
    }

    public function getRecordIdByDomain(string $zoneId, string $recordName)
    {
        $records = $this->request("GET", "/zones/$zoneId/dns_records?name=$recordName");

        if (!empty($records['result']) && is_array($records['result'])) {
            $result = [];
            foreach ($records['result'] as $res)
            {
                $result[] = $res['id'];
            }
            return $result;
        }
        return [];
    }

    public function getRecordIdByDomainAndIP(string $zoneId, string $recordName, string $ip)
    {
        $records = $this->request("GET", "/zones/$zoneId/dns_records?name=$recordName");

        if (!empty($records['result']) && is_array($records['result'])) {
            $result = [];
            foreach ($records['result'] as $res)
            {
                if ($res['content'] == $ip)
                    $result[] = $res['id'];
            }
            return $result;
        }
        return [];
    }

    public function sendData(
        string $name,
        string $content,
        string $type = 'A',
        int $ttl = 5, # auto
        bool $proxy = false
    ) {
        return ['name' => $name, 'type' => $type, 'content' => $content, 'proxied' => $proxy, 'comment' => 'updated from bot', 'ttl' => $ttl];
    }

    public function getMainDomain(string $host): string
    {
        $host = parse_url($host, PHP_URL_HOST) ?? $host;
        $host = preg_replace('/^www\./', '', $host);
        $parts = explode('.', $host);
        $count = count($parts);
        if ($count < 2) {
            return $host;
        }
        return $parts[$count - 2] . '.' . $parts[$count - 1];
    }

    public function createRecord(string $zoneId, array $data)
    {
        return $this->request("POST", "/zones/$zoneId/dns_records", $data);
    }

    public function updateRecord(string $zoneId, string $recordId, array $data)
    {
        return $this->request("PUT", "/zones/$zoneId/dns_records/$recordId", $data);
    }

    public function deleteRecord(string $zoneId, string $recordId)
    {
        return $this->request("DELETE", "/zones/$zoneId/dns_records/$recordId");
    }
}
